<?php

namespace ZGIT\NatayarkID\ConnectedAccount\Provider;

use XF;
use XF\Entity\ConnectedAccountProvider;
use XF\ConnectedAccount\Http\HttpResponseException;
use \XF\ConnectedAccount\Provider\AbstractProvider;

class NatayarkID extends AbstractProvider
{
	public function getOAuthServiceName()
	{
		return 'ZGIT\NatayarkID:Service\NatayarkID';
	}

	public function getProviderDataClass()
	{
		return 'ZGIT\NatayarkID:ProviderData\NatayarkID';
	}

	public function getDefaultOptions()
	{
		return [
			'client_id' => '',
			'client_secret' => ''
		];
	}

	public function getOAuthConfig(ConnectedAccountProvider $provider, $redirectUri = null)
	{
    $ClientSecretHash = password_hash($provider->options['client_secret'], PASSWORD_DEFAULT);
		return [
			'key' => $provider->options['client_id'],
			'secret' => $ClientSecretHash,
			'scopes' => ['basic'],
			'redirect' => $redirectUri ?: $this->getRedirectUri($provider)
		];
	}

	public function parseProviderError(HttpResponseException $e, &$error = null)
	{
		$errorDetails = json_decode($e->getResponseContent(), true);
		if (isset($errorDetails['error_description']))
		{
			$e->setMessage($errorDetails['error_description']);
		}
		parent::parseProviderError($e, $error);
	}

	public function getIconUrl()
	{
		return 'data:image/png;base64,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';
	}
}
